/*____________________________________________________________________________
        Copyright (C) 2000 Networks Associates Technology, Inc.
        All rights reserved.

	Wrapper functions for KeyDB library
	
        $Id: pgpClientKeyDB.c,v 1.14 2001/01/27 03:39:38 ajivsov Exp $
____________________________________________________________________________*/
#include <string.h>

#include "pgpMem.h"
#include "pgpContext.h"
#include "pgpOptionListPriv.h"
#include "pgpKeys.h"
#include "pgpKeyPriv.h"
#include "pgpOpaqueStructs.h"
#include "pgpErrors.h"


/*____________________________________________________________________________
	Generate a key
____________________________________________________________________________*/

	PGPError
PGPGenerateKey(
	PGPContextRef		context,
	PGPKeyDBObjRef		*key,
	PGPOptionListRef	firstOption,
	...
	)
{
	PGPError	error	= kPGPError_NoErr;
	
	pgpAssert( pgpContextIsValid( context ) );
	pgpAssert( IsntNull(key ) );
	
	if (IsntNull( key ) )
		*key	= kInvalidPGPKeyDBObjRef;
	
	pgpEnterPGPErrorFunction();

	if( pgpContextIsValid( context ) &&
			IsntNull(key ) )
	{
		va_list				args;
		PGPOptionListRef	optionList;

		va_start( args, firstOption );
		optionList = pgpBuildOptionListArgs( context,
			FALSE, firstOption, args );

		error = pgpGetOptionListError( optionList );
		if( IsntPGPError( error ) )
		{
			error = pgpGenerateKeyInternal( key, optionList );
		}
		va_end( args );

		PGPFreeOptionList( optionList );
	}
	else
	{
		va_list				args;
		
		va_start( args, firstOption );
		pgpFreeVarArgOptionList( firstOption, args);
		va_end( args );
		
		error = kPGPError_BadParams;
	}
	
	pgpAssertErrWithPtr( error, *key );
	
	return( error );
}


/*____________________________________________________________________________
	Generate a subkey
____________________________________________________________________________*/

	PGPError
PGPGenerateSubKey(
	PGPContextRef		context,
	PGPKeyDBObjRef		*subkey,
	PGPOptionListRef	firstOption,
	...
	)
{
	PGPError	error	= kPGPError_NoErr;
	
	pgpAssert( pgpContextIsValid( context ) );
	pgpAssertAddrValid( subkey, PGPKeyDBObjRef );
	
	if (IsntNull( subkey ) )
		*subkey	= kInvalidPGPKeyDBObjRef;
		
	pgpEnterPGPErrorFunction();

	if( pgpContextIsValid( context ) &&
		IsntNull( subkey ) )
	{
		va_list				args;
		PGPOptionListRef	optionList;
		
		*subkey = NULL;

		va_start( args, firstOption );
		optionList = pgpBuildOptionListArgs( context,
			FALSE, firstOption, args );

		error = pgpGetOptionListError( optionList );
		if( IsntPGPError( error ) )
		{
			error = pgpGenerateSubKeyInternal( subkey, optionList );
		}
		va_end( args );

		PGPFreeOptionList( optionList );
	}
	else
	{
		va_list				args;
		
		va_start( args, firstOption );
		pgpFreeVarArgOptionList( firstOption, args);
		va_end( args );
		
		error = kPGPError_BadParams;
	}
	
	return( error );
}


/*____________________________________________________________________________
	Certify (via a signature) new primary userid
____________________________________________________________________________*/

	PGPError
PGPSetPrimaryUserID (
	PGPKeyDBObjRef		userid,
	PGPOptionListRef	firstOption,
	...
	)
{
	PGPError	error;
	
	pgpAssertAddrValid( userid, PGPKeyDBObj );
	
	pgpEnterPGPErrorFunction();

	if( IsntNull( userid ) )
	{
		va_list				args;
		PGPOptionListRef	optionList;
		PGPContextRef		context;
		
		va_start( args, firstOption );
		context	= PGPPeekKeyDBObjContext( userid );
	
		optionList = pgpBuildOptionListArgs( context,
			FALSE, firstOption, args );

		error = pgpGetOptionListError( optionList );
		if( IsntPGPError( error ) )
		{
			error = pgpCertifyPrimaryUserIDInternal( userid, optionList );
		}
		va_end( args );

		PGPFreeOptionList( optionList );
	}
	else
	{
		va_list				args;
		
		va_start( args, firstOption );
		pgpFreeVarArgOptionList( firstOption, args);
		va_end( args );
		
		error = kPGPError_BadParams;
	}
	
	return( error );
}


/*____________________________________________________________________________
	Certify a userid, possibly with trust information
____________________________________________________________________________*/

	PGPError
PGPCertifyUserID(
	PGPKeyDBObjRef		userid,
	PGPKeyDBObjRef		certifyingKey,
	PGPOptionListRef	firstOption,
	...
	)
{
	PGPError	error;
	
	pgpAssertAddrValid( userid, PGPKeyDBObj );
	pgpAssert( pgpKeyIsValid( certifyingKey ) );
	
	pgpEnterPGPErrorFunction();

	if( IsntNull( userid ) && pgpKeyIsValid( certifyingKey ) )
	{
		va_list				args;
		PGPOptionListRef	optionList;
		PGPContextRef		context;
		
		va_start( args, firstOption );
		context	= PGPPeekKeyDBObjContext( certifyingKey );
	
		optionList = pgpBuildOptionListArgs( context,
			FALSE, firstOption, args );

		error = pgpGetOptionListError( optionList );
		if( IsntPGPError( error ) )
		{
			error = pgpCertifyUserIDInternal( userid, certifyingKey,
											  optionList );
		}
		va_end( args );

		PGPFreeOptionList( optionList );
	}
	else
	{
		va_list				args;
		
		va_start( args, firstOption );
		pgpFreeVarArgOptionList( firstOption, args);
		va_end( args );
		
		error = kPGPError_BadParams;
	}
	
	return( error );
}


/*____________________________________________________________________________
	Revoke a certificate
____________________________________________________________________________*/

	PGPError
PGPRevokeSig(
	PGPKeyDBObjRef		sig,
	PGPOptionListRef	firstOption,
	...
	)
{
	PGPError		error = kPGPError_NoErr;
	
	pgpAssert( pgpSigIsValid( sig ) );
	
	pgpEnterPGPErrorFunction();

	if( pgpSigIsValid( sig ) )
	{
		va_list				args;
		PGPOptionListRef	optionList;
		PGPContextRef		context;
	
		va_start( args, firstOption );
		context		= PGPPeekKeyDBObjContext( sig );
		optionList = pgpBuildOptionListArgs( context, FALSE,
											 firstOption, args );

		error = pgpGetOptionListError( optionList );
		if( IsntPGPError( error ) )
		{
			error = pgpRevokeSigInternal( sig, optionList );
		}
		va_end( args );

		PGPFreeOptionList( optionList );
	}
	else
	{
		va_list				args;

		va_start( args, firstOption );
		pgpFreeVarArgOptionList( firstOption, args);
		va_end( args );
		
		if( IsntPGPError( error ) )
		{
			error = kPGPError_BadParams;
		}
	}
	
	return( error );
}


/*____________________________________________________________________________
	Revoke a key or subkey
____________________________________________________________________________*/

	PGPError
PGPRevoke(
	PGPKeyDBObjRef		key,
	PGPOptionListRef	firstOption,
	...
	)
{
	PGPError		error;
	
	pgpAssert( pgpKeyIsValid( key ) || pgpSubKeyIsValid( key ) );
	
	pgpEnterPGPErrorFunction();

	if( pgpKeyIsValid( key )  ||  pgpSubKeyIsValid( key ) )
	{
		va_list				args;
		PGPOptionListRef	optionList;
		PGPContextRef		context;
		
		va_start( args, firstOption );
		context		= PGPPeekKeyDBObjContext( key );
		optionList = pgpBuildOptionListArgs( context, FALSE,
											 firstOption, args );

		error = pgpGetOptionListError( optionList );
		if( IsntPGPError( error ) )
		{
			if( pgpObjectType( key ) == RINGTYPE_KEY )
				error = pgpRevokeKeyInternal( key, optionList );
			else
				error = pgpRevokeSubKeyInternal( key, optionList );
		}
		va_end( args );

		PGPFreeOptionList( optionList );
	}
	else
	{
		va_list				args;
		
		va_start( args, firstOption );
		pgpFreeVarArgOptionList( firstOption, args);
		va_end( args );
		
		error = kPGPError_BadParams;
	}
	
	return( error );
}



/*____________________________________________________________________________
	Add a new userid to a key
____________________________________________________________________________*/

	PGPError
PGPAddUserID(
	PGPKeyDBObjRef		key,
	char const *		userID,
	PGPOptionListRef	firstOption,
	...
	)
{
	PGPError	error;
	
	pgpAssert( pgpKeyIsValid( key ) );
	pgpAssertAddrValid( userID, char );
	
	pgpEnterPGPErrorFunction();

	if( pgpKeyIsValid( key )  && IsntNull( userID )
		&& strlen( userID ) < 256 )
	{
		va_list				args;
		PGPOptionListRef	optionList;
		PGPContextRef		context;
		
		va_start( args, firstOption );
		
		context		= PGPPeekKeyDBObjContext( key );
		optionList = pgpBuildOptionListArgs( context, FALSE,
											 firstOption, args );

		error = pgpGetOptionListError( optionList );
		if( IsntPGPError( error ) )
		{
			error = pgpAddUserIDInternal( key, userID, optionList );
		}
		va_end( args );

		PGPFreeOptionList( optionList );
	}
	else
	{
		va_list				args;
		
		va_start( args, firstOption );
		pgpFreeVarArgOptionList( firstOption, args);
		va_end( args );
		
		error = kPGPError_BadParams;
	}
	
	return( error );
}


/*____________________________________________________________________________
	Add a generalized attribute to a key
____________________________________________________________________________*/

	PGPError
PGPAddAttributeUserID(
	PGPKeyDBObjRef		key,
	PGPAttributeType	attributeType,
	PGPByte			   *attributeData,
	PGPSize				attributeLength,
	PGPOptionListRef	firstOption,
	...
	)
{
	PGPError	error;
	
	pgpAssert( pgpKeyIsValid( key ) );
	pgpAssertAddrValid( attributeData, PGPByte );
	
	pgpEnterPGPErrorFunction();
	
	if( pgpKeyIsValid( key )  && IsntNull( attributeData ) )
	{
		va_list				args;
		PGPOptionListRef	optionList;
		PGPContextRef		context;
		
		va_start( args, firstOption );
		
		context		= PGPPeekKeyDBObjContext( key );
		optionList = pgpBuildOptionListArgs( context, FALSE,
											 firstOption, args );

		error = pgpGetOptionListError( optionList );
		if( IsntPGPError( error ) )
		{
			error = pgpAddAttributeInternal( key, attributeType,
							attributeData, attributeLength, optionList );
		}
		va_end( args );

		PGPFreeOptionList( optionList );
	}
	else
	{
		va_list				args;
		
		va_start( args, firstOption );
		pgpFreeVarArgOptionList( firstOption, args);
		va_end( args );
		
		error = kPGPError_BadParams;
	}
	
	return( error );
}


/*____________________________________________________________________________
	Find out how much entropy is needed
____________________________________________________________________________*/

	PGPUInt32
PGPGetKeyEntropyNeeded(
	PGPContextRef		context,
	PGPOptionListRef	firstOption,
	...
	)
{
	PGPError	error	= kPGPError_NoErr;
	PGPUInt32	result = ~(PGPUInt32)0;
	
	pgpAssert( pgpContextIsValid( context ) );
	
	pgpEnterPGPErrorFunction();

	if( pgpContextIsValid( context ) )
	{
		va_list				args;
		PGPOptionListRef	optionList;
		
		va_start( args, firstOption );
		optionList = pgpBuildOptionListArgs( context,
			FALSE, firstOption, args );

		error = pgpGetOptionListError( optionList );
		if( IsntPGPError( error ) )
		{
			result = pgpKeyEntropyNeededInternal( context, optionList );
		}
		va_end( args );

		PGPFreeOptionList( optionList );
	}
	else
	{
		va_list				args;
		
		va_start( args, firstOption );
		pgpFreeVarArgOptionList( firstOption, args);
		va_end( args );
	}
	
	/* No mechanism to return an error at this time */
	return( result );
}


/*____________________________________________________________________________
	Create a new X.509 certificate or self-certificate
____________________________________________________________________________*/

	PGPError
PGPCreateSelfSignedX509Certificate(
	PGPKeyDBObjRef		signingKey,
	PGPKeyDBObjRef		*newSig,
	PGPOptionListRef	firstOption,
	...
	)
{
	PGPError		error	= kPGPError_NoErr;
	
	pgpAssert( pgpKeyIsValid( signingKey ) );
	pgpAssert( IsntNull( newSig ) );
	
	if (IsntNull( newSig ) )
		*newSig	= kInvalidPGPKeyDBObjRef;
	
	pgpEnterPGPErrorFunction();

	if( pgpKeyIsValid( signingKey ) )
	{
		va_list				args;
		PGPContextRef		context;
		PGPOptionListRef	optionList;
	
		context = PGPPeekKeyDBObjContext( signingKey );
		va_start( args, firstOption );
		
		optionList = pgpBuildOptionListArgs( context,
			FALSE, firstOption, args );

		error = pgpGetOptionListError( optionList );
		if( IsntPGPError( error ) )
		{
			error = pgpCreateX509CertificateInternal( signingKey, signingKey,
													  newSig, optionList );
		}
		va_end( args );

		/* optionList is freed in internal function */
	}
	else
	{
		va_list				args;
		
		va_start( args, firstOption );
		pgpFreeVarArgOptionList( firstOption, args);
		va_end( args );
		
		error = kPGPError_BadParams;
	}
	
	return( error );
}


	PGPError
PGPCreateX509Certificate(
	PGPKeyDBObjRef		signingSig,
	PGPKeyDBObjRef		signedKey,
	PGPKeyDBObjRef		*newSig,
	PGPOptionListRef	firstOption,
	...
	)
{
	PGPError		error	= kPGPError_NoErr;
	
	pgpAssert( pgpSigIsValid( signingSig ) );
	pgpAssert( pgpKeyIsValid( signedKey ) );
	pgpAssert( IsntNull( newSig ) );
	
	if (IsntNull( newSig ) )
		*newSig	= kInvalidPGPKeyDBObjRef;
	
	pgpEnterPGPErrorFunction();

	if( pgpSigIsValid( signingSig ) && pgpKeyIsValid( signedKey ) )
	{
		va_list				args;
		PGPContextRef		context;
		PGPOptionListRef	optionList;
	
		context = PGPPeekKeyDBObjContext( signingSig );
		va_start( args, firstOption );
		
		optionList = pgpBuildOptionListArgs( context,
			FALSE, firstOption, args );

		error = pgpGetOptionListError( optionList );
		if( IsntPGPError( error ) )
		{
			error = pgpCreateX509CertificateInternal( signingSig, signedKey,
													  newSig, optionList );
		}
		va_end( args );

		/* optionList is freed in internal function */
	}
	else
	{
		va_list				args;
		
		va_start( args, firstOption );
		pgpFreeVarArgOptionList( firstOption, args);
		va_end( args );
		
		error = kPGPError_BadParams;
	}
	
	return( error );
}

/*____________________________________________________________________________
	Export a key, keyset, or subset of a key to a buffer or file
____________________________________________________________________________*/

	PGPError
PGPExport(
	PGPContextRef		context,
	PGPOptionListRef	firstOption,
	...
	)
{
	PGPError		error	= kPGPError_NoErr;
	
	pgpAssert( pgpContextIsValid( context ) );
	
	pgpEnterPGPErrorFunction();

	if( pgpContextIsValid( context ) )
	{
		va_list				args;
		PGPOptionListRef	optionList;
	
		va_start( args, firstOption );
		
		optionList = pgpBuildOptionListArgs( context,
			FALSE, firstOption, args );

		error = pgpGetOptionListError( optionList );
		if( IsntPGPError( error ) )
		{
			error = pgpExportInternal( context, optionList );
		}
		va_end( args );

		/* optionList is freed in internal function */
	}
	else
	{
		va_list				args;
		
		va_start( args, firstOption );
		pgpFreeVarArgOptionList( firstOption, args);
		va_end( args );
		
		error = kPGPError_BadParams;
	}
	
	return( error );
}

/*____________________________________________________________________________
	Import a keyset from a buffer or file
____________________________________________________________________________*/

	PGPError
PGPImport(
	PGPContextRef		context,
	PGPKeyDBRef			*keys,
	PGPOptionListRef	firstOption,
	...
	)
{
	PGPError	error	= kPGPError_NoErr;
	
	if( IsntNull( keys ) )
		*keys	= kInvalidPGPKeyDBRef;
		
	pgpEnterPGPErrorFunction();

	if( pgpContextIsValid( context ) && IsntNull( keys ) )
	{
		va_list				args;
		PGPOptionListRef	optionList;
		
		va_start( args, firstOption );
		optionList = pgpBuildOptionListArgs( context,
			FALSE, firstOption, args );

		error = pgpGetOptionListError( optionList );
		if( IsntPGPError( error ) )
		{
			error = pgpImportKeySetInternal( context, keys, optionList );
		}
		va_end( args );

		/* optionList is freed in internal function */
	}
	else
	{
		va_list				args;
		
		va_start( args, firstOption );
		pgpFreeVarArgOptionList( firstOption, args);
		va_end( args );
		
		error = kPGPError_BadParams;
	}
	
	return( error );
}

/*____________________________________________________________________________
	Change (top level) key passphrase
____________________________________________________________________________*/

	PGPError
PGPChangePassphrase(
	PGPKeyDBObjRef		key,
	PGPOptionListRef	firstOption,
	...
	)
{
	PGPError	error	= kPGPError_NoErr;
	
	pgpEnterPGPErrorFunction();

	pgpAssert( pgpKeyIsValid( key ) || pgpSubKeyIsValid( key ) );
	
	if( pgpKeyIsValid( key )  ||  pgpSubKeyIsValid( key ) )
	{
		va_list				args;
		PGPContextRef		context;
		PGPOptionListRef	optionList;

		context = PGPPeekKeyDBObjContext( key );

		va_start( args, firstOption );
		optionList = pgpBuildOptionListArgs( context,
			FALSE, firstOption, args );

		error = pgpGetOptionListError( optionList );
		if( IsntPGPError( error ) )
		{
			if( pgpObjectType( key ) == RINGTYPE_KEY )
				error = pgpChangePassphraseInternal( key, optionList );
			else
				error = pgpChangeSubKeyPassphraseInternal( key, optionList );
		}
		va_end( args );

		PGPFreeOptionList( optionList );
	}
	else
	{
		va_list				args;
		
		va_start( args, firstOption );
		pgpFreeVarArgOptionList( firstOption, args);
		va_end( args );
		
		error = kPGPError_BadParams;
	}
	
	return( error );
}

/*____________________________________________________________________________
	Check passphrase validity
____________________________________________________________________________*/

	PGPBoolean
PGPPassphraseIsValid(
	PGPKeyDBObjRef		key,
	PGPOptionListRef	firstOption,
	...
	)
{
	PGPBoolean	rslt	= FALSE;
	PGPError	error	= kPGPError_NoErr;
	
	pgpAssert( pgpKeyIsValid( key ) );
	
	pgpEnterBooleanFunction( FALSE );

	if( pgpKeyIsValid( key ) )
	{
		va_list				args;
		PGPContextRef		context;
		PGPOptionListRef	optionList;

		context = PGPPeekKeyDBObjContext( key );

		va_start( args, firstOption );
		optionList = pgpBuildOptionListArgs( context,
			FALSE, firstOption, args );

		error = pgpGetOptionListError( optionList );
		if( IsntPGPError( error ) )
		{
			rslt = pgpPassphraseIsValidInternal( key, optionList );
		}
		va_end( args );

		PGPFreeOptionList( optionList );

		return rslt;
	}
	else
	{
		va_list				args;
		
		va_start( args, firstOption );
		pgpFreeVarArgOptionList( firstOption, args);
		va_end( args );
		
		return FALSE;
	}
}

/*____________________________________________________________________________
	Set key axiomatic, possibly testing passphrase
____________________________________________________________________________*/

	PGPError
PGPSetKeyAxiomatic(
	PGPKeyDBObjRef		key,
	PGPBoolean			setAxiomatic,
	PGPOptionListRef	firstOption,
	...
	)
{
	PGPError	error	= kPGPError_NoErr;
	
	pgpAssert( pgpKeyIsValid( key ) );
	
	pgpEnterPGPErrorFunction();

	if( pgpKeyIsValid( key ) )
	{
		va_list				args;
		PGPContextRef		context;
		PGPOptionListRef	optionList;

		context = PGPPeekKeyDBObjContext( key );

		va_start( args, firstOption );
		optionList = pgpBuildOptionListArgs( context,
			FALSE, firstOption, args );

		error = pgpGetOptionListError( optionList );
		if( IsntPGPError( error ) )
		{
			error = pgpSetKeyAxiomaticInternal( key, setAxiomatic, optionList);
		}
		va_end( args );

		PGPFreeOptionList( optionList );
	}
	else
	{
		va_list				args;
		
		va_start( args, firstOption );
		pgpFreeVarArgOptionList( firstOption, args);
		va_end( args );
		
		error = kPGPError_BadParams;
	}
	
	return( error );
}


/*____________________________________________________________________________
	Get the passkeybuffer to unlock a key, given its passphrase
____________________________________________________________________________*/

	PGPError
PGPGetPasskeyBuffer(
	PGPKeyDBObjRef		key,
	void			   *passkeyBuffer,
	PGPOptionListRef	firstOption,
	...
	)
{
	PGPError	error	= kPGPError_NoErr;
	
	pgpAssert( pgpKeyIsValid( key ) || pgpSubKeyIsValid( key ) );
	pgpAssert( IsntNull( passkeyBuffer ) );
	
	pgpEnterPGPErrorFunction();

	if( IsntNull( passkeyBuffer ) &&
		( pgpKeyIsValid( key ) || pgpSubKeyIsValid( key ) )  )
	{
		va_list				args;
		PGPContextRef		context;
		PGPOptionListRef	optionList;

		context = PGPPeekKeyDBObjContext( key );

		va_start( args, firstOption );
		optionList = pgpBuildOptionListArgs( context,
			FALSE, firstOption, args );

		error = pgpGetOptionListError( optionList );
		if( IsntPGPError( error ) )
		{
			if( pgpObjectType( key ) == RINGTYPE_KEY )
				error = pgpGetKeyPasskeyBufferInternal( key, passkeyBuffer,
														optionList );
			else
				error = pgpGetSubKeyPasskeyBufferInternal( key, passkeyBuffer,
														   optionList );
		}
		va_end( args );

		PGPFreeOptionList( optionList );
	}
	else
	{
		va_list				args;
		
		va_start( args, firstOption );
		pgpFreeVarArgOptionList( firstOption, args);
		va_end( args );
		
		error = kPGPError_BadParams;
	}
	
	return( error );
}


/*____________________________________________________________________________
	Add, Remove, or Replace key self-sig options
____________________________________________________________________________*/

	PGPError
PGPAddKeyOptions(
	PGPKeyDBObjRef		key,
	PGPOptionListRef	firstOption,
	...
	)
{
	PGPError	error	= kPGPError_NoErr;
	
	pgpEnterPGPErrorFunction();

	pgpAssert( pgpKeyIsValid( key ) );
	
	if( pgpKeyIsValid( key ) )
	{
		va_list				args;
		PGPContextRef		context;
		PGPOptionListRef	optionList;

		context = PGPPeekKeyDBObjContext( key );

		va_start( args, firstOption );
		optionList = pgpBuildOptionListArgs( context,
			FALSE, firstOption, args );

		error = pgpGetOptionListError( optionList );
		if( IsntPGPError( error ) )
		{
			error = pgpAddKeyOptionsInternal( key, optionList );
		}
		va_end( args );

		PGPFreeOptionList( optionList );
	}
	else
	{
		va_list				args;
		
		va_start( args, firstOption );
		pgpFreeVarArgOptionList( firstOption, args);
		va_end( args );
		
		error = kPGPError_BadParams;
	}
	
	return( error );
}

	PGPError
PGPRemoveKeyOptions(
	PGPKeyDBObjRef		key,
	PGPOptionListRef	firstOption,
	...
	)
{
	PGPError	error	= kPGPError_NoErr;
	
	pgpAssert( pgpKeyIsValid( key ) );
	
	pgpEnterPGPErrorFunction();

	if( pgpKeyIsValid( key ) )
	{
		va_list				args;
		PGPContextRef		context;
		PGPOptionListRef	optionList;

		context = PGPPeekKeyDBObjContext( key );

		va_start( args, firstOption );
		optionList = pgpBuildOptionListArgs( context,
			FALSE, firstOption, args );

		error = pgpGetOptionListError( optionList );
		if( IsntPGPError( error ) )
		{
			error = pgpRemoveKeyOptionsInternal( key, optionList );
		}
		va_end( args );

		PGPFreeOptionList( optionList );
	}
	else
	{
		va_list				args;
		
		va_start( args, firstOption );
		pgpFreeVarArgOptionList( firstOption, args);
		va_end( args );
		
		error = kPGPError_BadParams;
	}
	
	return( error );
}

	PGPError
PGPUpdateKeyOptions(
	PGPKeyDBObjRef		key,
	PGPOptionListRef	firstOption,
	...
	)
{
	PGPError	error	= kPGPError_NoErr;
	
	pgpAssert( pgpKeyIsValid( key ) );
	
	pgpEnterPGPErrorFunction();

	if( pgpKeyIsValid( key ) )
	{
		va_list				args;
		PGPContextRef		context;
		PGPOptionListRef	optionList;

		context = PGPPeekKeyDBObjContext( key );

		va_start( args, firstOption );
		optionList = pgpBuildOptionListArgs( context,
			FALSE, firstOption, args );

		error = pgpGetOptionListError( optionList );
		if( IsntPGPError( error ) )
		{
			error = pgpUpdateKeyOptionsInternal( key, optionList );
		}
		va_end( args );

		PGPFreeOptionList( optionList );
	}
	else
	{
		va_list				args;
		
		va_start( args, firstOption );
		pgpFreeVarArgOptionList( firstOption, args);
		va_end( args );
		
		error = kPGPError_BadParams;
	}
	
	return( error );
}


/*__Editor_settings____

	Local Variables:
	tab-width: 4
	End:
	vi: ts=4 sw=4
	vim: si
_____________________*/
